/*
  FUSE: Filesystem in Userspace
  ACV - IST -  Taguspark, October 2012

*/

#define FUSE_USE_VERSION 26

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fuse.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <sys/time.h>

#ifdef HAVE_SETXATTR
#include <sys/xattr.h>
#endif

#include "block.h"
#include "fs.h"

#define BLOCK_SIZE 512
#ifndef NUM_BLOCKS
// default storage of 2 MB (8*512 blocks * 512 bytes/block)
#define NUM_BLOCKS (8*512)
#endif

// maximum amount of directory entries 
#define MAX_READDIR_ENTRIES 64 
// maximum size of a file name used in barefs
#define MAX_FILE_NAME_SIZE 14


///////////////////////////////////////////////////////////

static fs_t* FS;

///////////////////////////////////////////////////////////
//
// Prototypes for all these functions, and the C-style comments,
// come indirectly from /usr/include/fuse.h
////////////////////////////////////////////////////////////

/**
 * Initialize filesystem
 */
void *barefs_init(struct fuse_conn_info *conn)
{
    FS = fs_new(NUM_BLOCKS);
    fs_format(FS);
    return NULL;
}

/**
 * Clean up filesystem
 *
 * Called on filesystem exit.
 *
 */
void barefs_destroy(void *userdata)
{
}


/**
 * Create and open a file
 *
 * If the file does not exist, first create it with the specified
 * mode, and then open it.
 * 
 * NOTICE: In this implementation, files are created only in root directory
 * TODO: Implement the creation of files in any subdirectory!!!!!!!
 */
int barefs_create(const char *path, mode_t mp, struct fuse_file_info *fi) 
{

   int res = -1;
   inodeid_t dir = 1 ; // 1 is the root directory inode number.
   inodeid_t fileid;
   char* pathname="/";  // slash character "/" identifies the root directory
 
   if (fs_lookup(FS,pathname,&dir) == 1) {
	   if (!fs_create(FS,dir,(char*)path,&fileid)) {
		 fi->fh = fileid;
		 res = 0;
	   }
   }     

   return res;

}



/** Get file attributes.
 *
 * TODO: File attributes must comphreend the number of a file hardlinks
*/
int barefs_getattr(const char *path, struct stat *stbuf)
{

   int res = -ENOENT;
   inodeid_t fileid;

   memset(stbuf, 0, sizeof(struct stat));
   
   // Root Directory Attributes
   if ( strcmp((char*)path,"/")== 0 ) {
		stbuf->st_mode = S_IFDIR | 0777;
		stbuf->st_nlink = 2;
		stbuf->st_size = BLOCK_SIZE;
		return 0;		
   }
  
   if (fs_lookup(FS,(char*)path,&fileid) == 1) {
    	//printf("[barefs_getattr] filename: '%s' [inode: %d]\n", path, fileid);
      	fs_file_attrs_t attrs;
      	if (fs_get_attrs(FS,fileid,&attrs) == 0) {
	      if (attrs.type == 2) {
                        stbuf->st_nlink = 1;
			stbuf->st_mode = S_IFREG | 0777;
			stbuf->st_size = attrs.size;
			res = 0;
              } else if (attrs.type == 1) {
				stbuf->st_nlink = 2;
				stbuf->st_mode = S_IFDIR | 0777;
				stbuf->st_size = BLOCK_SIZE;
				res = 0;
	             }
      	}
   } 
   return res;   
}

/** Read directory
 *
*/ 
int barefs_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
		       off_t offset, struct fuse_file_info *fi)
{
    int res = -ENOENT;	
    (void) offset;
    (void) fi;
    inodeid_t fileid;
    int maxentries;	
    fs_file_attrs_t attrs;

    if(strcmp(path, "/") == 0) 
	   fileid=1; //1 is the root directory inode number
    else { 
          if (fs_lookup(FS,(char*)path,&fileid) != 1) 
	      return res;
    }

    filler(buf, ".", NULL, 0);
    filler(buf, "..", NULL, 0);
	
    if (fs_get_attrs(FS,fileid,&attrs) == 0) {
	if (attrs.type == 1) {
		maxentries = attrs.num_entries;
		//printf("[barefs_readdir] Directory '%s' has %d entries.\n", path, attrs.num_entries);			
		res = 0;
         }
		
	if (maxentries) {
		// Read the directory
		fs_file_name_t entries[MAX_READDIR_ENTRIES];
		int numentries;
		if (!fs_readdir(FS,fileid,entries,maxentries,&numentries)) {
		    int i;
	            for (i = 0; i < numentries; i++) {						
			filler(buf, entries[i].name, NULL, 0);  
		    }
		    res = 0;	  
		}
	}	
    }
    return res;	
}


/**Create a directory with the given name.
* 
*/
int barefs_mkdir(const char* path, mode_t mode)
{
   int res = -ENOENT;

// TODO: IMPLEMENT !!!!!!   

   return res;	
}

/**Remove the given directory.
*
* This should succeed only if the directory is empty
*/
int barefs_rmdir(const char* path)
{
   int res = -ENOENT;

// TODO: IMPLEMENT !!!!!!

   return res;	
}


/**Create hard link
*
* Create a hard link between "from" and "to".
*/
int barefs_link(const char* from, const char* to)
{
   int res = -ENOENT;

// TODO: IMPLEMENT !!!!!!

   return res;	
}



/** Remove (delete) the given file or hard link 
*
* Note that unlink only deletes the data when the last hard link is removed.
*
* NOTICE: In this implementation, files (or hard links) are removed only in root directory
* TODO: Implement the removing of files (or hard links) in any subdirectory!!!!!!!
*
*/
int barefs_unlink(const char* path)
{
   int res = -ENOENT;
   inodeid_t fileid, dir;
   dir=1; //1 is the root dir inode number 
   char* pathname="/";  // the slash character "/" identifies the root directory 

   if (fs_lookup(FS,pathname,&dir) == 1) {
	if (fs_remove(FS,dir,(char*) path,&fileid) == 0) {
	    	//printf("[barefs_unlink] Removing file '%s' \n", path);
        	res=0;
      	}
   } 

   return res;	
}


/** File open operation
 *
 * No creation, or truncation flags (O_CREAT, O_EXCL, O_TRUNC)
 * will be passed to open().  Open should check if the operation
 * is permitted for the given flags.  Optionally open may also
 * return an arbitrary filehandle in the fuse_file_info structure,
 * which will be passed to all file operations.
 *
 */
int barefs_open(const char *path, struct fuse_file_info *fi)
{
   int res = -ENOENT;
   inodeid_t fileid;

   if (fs_lookup(FS,(char*)path,&fileid) == 1) {
	fi->fh= fileid;
	res = 0;
   } 

   return res;	
}

/** Read data from an open file
 *
 * Read should return exactly the number of bytes requested except
 * on EOF or error, otherwise the rest of the data will be
 * substituted with zeroes (check fuse documentation for exceptions).
 *
 */
int barefs_read(const char *path, char *buf, size_t size, off_t offset,
		    struct fuse_file_info *fi)
{
   int res = -1;
   int fileid = fi->fh;
   int nread = 0; 
   
   if (fs_read(FS,fileid,offset,size,buf,&nread) == 0){
	offset += nread;
	res = nread;
   }   
   
   return res;	

}


/** Write data to an open file
 *
 * Write should return exactly the number of bytes requested
 * except on error Read data from an open file
 *
 * Read should return exactly the number of bytes requested except
 * on EOF or error, otherwise the rest of the data will be
 * substituted with zeroes (check fuse documentation for exceptions).
 *
 */
int barefs_write(const char *path, const char *buf, size_t size,
		     off_t offset, struct fuse_file_info *fi)
{
   int res=-1;
   int fileid = fi->fh; 

   if (!fs_write(FS,fileid,offset,size,(char*)buf)){
		res=(int) size;
   }   

   return res;	
}


/** Create a file node
 *
*/ 
// mknod() apparently skould be called, instead of create(),
// for many shell commands performing file creation, directory 
// listing, etc ..
int barefs_mknod(const char *path, mode_t mode, dev_t dev)
{
   return 0;
}


/*
* Flush is called on each close() of a file descriptor.  So if a
 * filesystem wants to return write errors in close() and the file
 * has cached dirty data, this is a good place to write back data
 * and return any errors.
 */
int barefs_flush(const char *path, struct fuse_file_info *fi)
{   
   return 0;
}

/** Change the access and/or modification times of a file */
int barefs_utime(const char *path, struct utimbuf *ubuf)
{
    return 0;
}

/** Change the owner and group of a file */
int barefs_chown(const char *path, uid_t uid, gid_t gid)  
{ 
    return 0;
}

/** Change the permission bits of a file */
int barefs_chmod(const char *path, mode_t mode)
{  
    return 0;
}

/** Change the size of a file */
int barefs_truncate(const char *path, off_t newsize)
{
   int res = -ENOENT;
   inodeid_t fileid;

   if (fs_lookup(FS,(char*)path,&fileid) == 1) {
        if (fs_truncate(FS, fileid) == 0)
	   res = 0;
   } 
	
   return res;
}

/** Release an open file
 *
 * Release is called when there are no more references to an open
 * file: all file descriptors are closed and all memory mappings
 * are unmapped.
 *
 * For every open() call there will be exactly one release() call
 * with the same flags and file descriptor.  It is possible to
 * have a file opened more than once, in which case only the last
 * release will mean, that no more reads/writes will happen on the
 * file.  The return value of release is ignored.
 *
 */
int barefs_release(const char *path, struct fuse_file_info *fi)
{
   return 0;
}

/** Get extended attributes */
// Is usefull when executing "ls" shell command
int barefs_getxattr(const char *path, const char *name, char *value, size_t size)
{  
   return 0;
}

static struct fuse_operations barefs_oper = {
	.getattr	= barefs_getattr,
	.readdir	= barefs_readdir,
	.open		= barefs_open,
	.create		= barefs_create,
	.mkdir		= barefs_mkdir,
	.rmdir		= barefs_rmdir,
	.link		= barefs_link,
	.unlink		= barefs_unlink,  
	.mknod		= barefs_mknod,
	.read		= barefs_read,
	.write		= barefs_write,
	.init		= barefs_init,
	.flush		= barefs_flush,
	.destroy	= barefs_destroy,
	.release	= barefs_release,
	.utime		= barefs_utime,
	.chown		= barefs_chown,
	.chmod		= barefs_chmod,
	.truncate	= barefs_truncate,
	.getxattr 	= barefs_getxattr,	
	
};

int main(int argc, char *argv[])
{
    return fuse_main(argc, argv, &barefs_oper, NULL);	
}


